/**
 * dtags-c:
 * @title: dtags-c
 * @subtitle: dtags-c
 * @project: dtags
 * @lang: fr,en
 * @authors: Philippe Roy <ph_roy@toutdoux.org>
 * @copyright: Copyright (c) 2001 Philippe Roy
 * @license: GNU GPL
 *
 * fr: Extraction de la documentation  partir de source en language C
 *
 * en: Extract documentation from C source file
 **/

/*
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <fcntl.h>
#include <parser.h>
#include <tree.h>
/*  #include <libxml/parser.h> */
/*  #include <libxml/tree.h> */

#include "dtags-c.h"

/*****************************************************************************/
/*** Standardisation du vocabulaire - Vocabulary standardization */
/*****************************************************************************/

#define XmlDoc xmlDoc
#define XmlNs xmlNs
#define XmlNode xmlNode
#define XmlAttribute xmlAttribute
#define XmlAttr xmlAttr
#define XmlChar xmlChar

/*****************************************************************************/
/*** Variables globales - Global variables */
/*****************************************************************************/

#define BUFFER_SIZE 256
#define MAX_RING 100
gchar *PACKAGE_COMMAND;
TdPopt popts[4];
GHashTable *tags_file;
GHashTable *tags_text;
GHashTable *tags_type;
GHashTable *tags_line;

/*****************************************************************************/
/*** Communs - Commons
/*****************************************************************************/

/**
 * td_hashtable_get_keys:
 *
 * fr: Retourne les cls
 *
 * en: Returns the keys
 *
 * Return value: list
 **/

GList* hashtable_get_keys (GHashTable *hashtable)
{
  GList *ret = NULL;
  g_hash_table_foreach (hashtable, (GHFunc) hashtable_get_keys_item, &ret);
  return ret;
}

void hashtable_get_keys_item (gpointer key, gpointer value, GList **list)
{
  *list = g_list_append (*list, g_strdup (key));
}

/**
 * td_hashtable_get_keys_type:
 *
 * fr: Retourne les cls avec le du type
 *
 * en: Returns the keys
 *
 * Return value: list
 **/

GList* hashtable_get_keys_type (gchar *type)
{
  GList *ret = NULL;
  if (!strcmp (type, "macro"))
    g_hash_table_foreach (tags_type, (GHFunc) hashtable_get_keys_macro, &ret);
  if (!strcmp (type, "enumerator"))
    g_hash_table_foreach (tags_type, (GHFunc) hashtable_get_keys_enumerator, &ret);
  if (!strcmp (type, "typedef"))
    g_hash_table_foreach (tags_type, (GHFunc) hashtable_get_keys_typedef, &ret);
  if (!strcmp (type, "struct"))
    g_hash_table_foreach (tags_type, (GHFunc) hashtable_get_keys_struct, &ret);
  if (!strcmp (type, "function"))
    g_hash_table_foreach (tags_type, (GHFunc) hashtable_get_keys_function, &ret);
  return ret;
}

void hashtable_get_keys_macro (gpointer key, gpointer value, GList **list)
{
  if (!strcmp (value, "macro"))
    *list = g_list_append (*list, g_strdup (key));
}

void hashtable_get_keys_enumerator (gpointer key, gpointer value, GList **list)
{
  if (!strcmp (value, "enumerator"))
    *list = g_list_append (*list, g_strdup (key));
}

void hashtable_get_keys_typedef (gpointer key, gpointer value, GList **list)
{
  if (!strcmp (value, "typedef"))
    *list = g_list_append (*list, g_strdup (key));
}

void hashtable_get_keys_struct (gpointer key, gpointer value, GList **list)
{
  if (!strcmp (value, "struct"))
    *list = g_list_append (*list, g_strdup (key));
}

void hashtable_get_keys_function (gpointer key, gpointer value, GList **list)
{
  if (!strcmp (value, "function"))
    *list = g_list_append (*list, g_strdup (key));
}

/**
 * td_hashtable_get_values:
 *
 * fr: Retourne les valeurs
 *
 * en: Returns the values
 *
 * Return value: list
 **/

GList* hashtable_get_values (GHashTable *hashtable)
{
  GList *ret = NULL;
  g_hash_table_foreach (hashtable, (GHFunc) hashtable_get_values_item, &ret);
  return ret;
}

void hashtable_get_values_item (gpointer key, gpointer value, GList **list)
{
  *list = g_list_append (*list, g_strdup (value));
}

/**
 * strsplit:
 * @string: string
 * @delimiter: delimiter
 * @index: index
 *
 * fr: Retourne la valeur[index] du tableau
 *
 * en: Returns the value[index] of the array
 *
 * Return value: value
 **/

gchar* strsplit (gchar *string, gchar *delimiter, gint index)
{
  gchar **string2;
  if (!string)
    return NULL;
  string2 = g_strsplit (string, delimiter, index+1);
  return (g_strstrip (string2 [index]));
}

/**
 * string_file:
 * @file: file
 *
 * fr: Retourne le contenu du fichier
 *
 * en: Returns the file content
 *
 * Return value: string
 **/

GString* string_file (gchar *file)
{
  gint i;
  gint fd;
  GString *ret;
  GIOChannel *iochannel;
  char *buff;
  guint readden;
  buff = g_malloc0 (BUFFER_SIZE);
  ret = g_string_new ("");
  fd = open (file, O_RDONLY);
  if (fd < 0)
    g_error ("dtags-c: loading file '%s' unreadable", file);
  iochannel = g_io_channel_unix_new (fd);
  g_io_channel_read (iochannel, buff, BUFFER_SIZE, &readden);
  if (readden == 0)
    return NULL;
  i=0;
  while (readden != 0)
    {
      g_string_append (ret, g_strdup (buff));
      g_free (buff);
      buff = g_malloc0 (BUFFER_SIZE);
      g_io_channel_read (iochannel, buff, BUFFER_SIZE, &readden);
    }
  close (fd);
  g_free (buff);
  return ret;
}

/**
 * string_replace:
 * @source: source
 * @before: before
 * @after: after
 *
 * fr: Remplace une chane de caractre
 *
 * en: Remplaces a string
 *
 * Return value: result
 **/

gchar* string_replace (gchar *source, gchar *before, gchar *after)
{
  gchar *txt_tmp;
  gchar *txt_tmp2;
  gint l, l1, l2, l3; 
  if ((!source) || (!before) || (!after))
    return NULL;
  txt_tmp = strstr (source, before);
  if (!txt_tmp)
    return source;
  l = strlen (source);
  l2 = strlen (txt_tmp);
  l1 = l-l2;
  l3 = strlen (before);
  txt_tmp2 = g_strdup_printf ("%s%s%s", g_strndup (source, l1), after, g_strndup (source+l1+l3, l2-l3));
  return txt_tmp2;
}

/*****************************************************************************/
/*** Arguments de la ligne de commande - Command line arguments */
/*****************************************************************************/

/**
 * popt_def:
 * @key: char key
 * @long key: string key
 * @argument: argument
 * @description: description
 *
 * fr: Cr une option de ligne de commande
 *
 * en: Creates an option of command line
 *
 * Return value: popt structure
 **/

TdPopt popt_def (gchar *key, gchar *longkey, gchar *argument, gchar *description)
{
  TdPopt ret = NULL;
  ret = g_malloc0 (sizeof (TdPopt_tmp));
  ret->key = g_strdup (key);
  ret->longkey = g_strdup (longkey);
  ret->argument = g_strdup (argument);
  ret->description = description;
  ret->flag = FALSE;
  ret->value = NULL;
  ret->nbpredef = 0;
  return ret;
}

/**
 * popts_help:
 * @nbpopts: counter of options of commande line
 * @popts[]: arraw of options of commande line options
 *
 * fr: Affiche l'usage
 *
 * en: Displays the usage
 **/

void popts_help (gint nbpopts, TdPopt popts[])
{
  gint i, j, k, key, longkey, argument;
  gint length = 0;

  /*** Alignement - Aligment */
  g_print ("%s [option ...]\n", PACKAGE_COMMAND);
  for (i=0; i<nbpopts; i++)
    {
      key = 0;
      longkey = 0;
      argument = 0;
      if (popts[i]->key)
	key = 3;
      if (popts[i]->longkey)
	longkey = strlen (popts[i]->longkey);
      if (popts[i]->argument)
	argument = strlen (popts[i]->argument);
      if (length < (9+key+longkey+argument))
	length = 9+key+longkey+argument;
    }


  /*** Texte - Text */
  for (i=0; i<nbpopts; i++)
    {
      g_print ("  ");
      key = 0;
      longkey = 0;
      argument = 0;
      if (popts[i]->key)
	{
	  g_print ("-%s, ", popts[i]->key);
	  key = 4;
	}
      if (popts[i]->longkey)
	{
	  g_print ("--%s ", popts[i]->longkey);
	  longkey = strlen (popts[i]->longkey);
	}
      if (popts[i]->argument)
	{
	  g_print ("%s", popts[i]->argument);
	  argument = strlen (popts[i]->argument);
	}
      for (j=7+key+longkey+argument; j<length; j++)
	g_print (" ");
      g_print ("%s\n", popts[i]->description);
      for (j=0; j<popts[i]->nbpredef; j++)
	{
	  for (k=0; k<length; k++)
	    g_print (" ");
	  g_print ("%s\n", popts[i]->predef[j]);
	}
    }
  exit (0);
}

/**
 * popts_parse:
 * @argc: counter of options used of command line
 * @argv[]: arraw of options used of command line
 * @nbpopts: counter of options of command line
 * @popts[]: arraw of options of command line
 *
 * fr: Analyse de la ligne de commande
 *
 * en: Parses the command line
 **/

void popts_parse (gint argc, char *argv[], gint nbpopts, TdPopt popts[])
{
  gint i, j;
  GList *option = NULL;
  GList *argument = NULL;
  gchar *txt_tmp = NULL;
  gchar *txt_tmp1 = NULL;
  gchar *txt_tmp2 = NULL;
  gboolean bool_tmp;

  /*** Dcoupage - Cutting */
  for (i=1; i<argc; i++)
    if (argv[i][0] == '-')
      {
	if (argv[i][1] == '-')
	  option = g_list_append (option, g_strdup (argv[i]+2));
	else
	  option = g_list_append (option, g_strdup (argv[i]+1));
	if (i!=1)
	  {
	    argument = g_list_append (argument, g_strdup (txt_tmp));
	    txt_tmp = "(null)";
	  }
      }
    else
      txt_tmp = g_strdup (argv[i]);
  if (i!=1)
    argument = g_list_append (argument, g_strdup (txt_tmp));

  /*** Affectation - Allocation */
  for (i=0; i<g_list_length (option); i++)
    {

      /*** Aide - Help */
      if ((!strcmp ("h", g_list_nth_data (option, i))) || (!strcmp ("help", g_list_nth_data (option, i))))
	{
	  popts_help (nbpopts, popts);
	  break;
	}
      if (!strcmp ("about", g_list_nth_data (option, i)))
	{
	  popts_about();
	  break;
	}
      if ((!strcmp ("v", g_list_nth_data (option, i))) || (!strcmp ("version", g_list_nth_data (option, i))))
	{
	  popts_version();
	  break;
	}

      /*** Application */
      bool_tmp = FALSE;
      for (j=0; j<nbpopts; j++)
	if (((popts[j]->key) && (!strcmp (popts[j]->key, g_list_nth_data (option, i)))) || 
	    ((popts[j]->longkey) && (!strcmp (popts[j]->longkey, g_list_nth_data (option, i)))))
	  {
	    popts[j]->flag = TRUE;
	    bool_tmp = TRUE;
	    if (g_list_nth_data (argument, i))
	      popts[j]->value = g_strdup (g_list_nth_data (argument, i));
	    else
	      if (popts[j]->argument)
		g_error ("dtags-c: option '%s' needs argument '%s'", (gchar*) g_list_nth_data (option, i), popts[j]->argument);
	  }
      if (!bool_tmp)
	g_error ("dtags-c: option '%s' unknow", (gchar*) g_list_nth_data (option, i));
    }
}

/**
 * popts_version:
 *
 * fr: Affiche la version du programme
 *
 * en: Display the version of the program
 **/

void popts_version (void)
{
  g_print ("%s %s\n", PACKAGE_COMMAND, DTAGS_VERSION);
  exit (0);
}

/*****************************************************************************/
/*** Traduction - Translation */
/*****************************************************************************/

/**
 * load_tags:
 * @file_in: input file location
 *
 * fr: Tags reading
 *
 * en: Lecture des tags
 **/

void load_tags (gchar *file_in)
{
  XmlDoc *doc;
  XmlNode *cur;
  gchar *name;
  tags_file = g_hash_table_new (g_str_hash, g_str_equal);
  tags_text = g_hash_table_new (g_str_hash, g_str_equal);
  tags_type = g_hash_table_new (g_str_hash, g_str_equal);
  tags_line = g_hash_table_new (g_str_hash, g_str_equal);
  xmlKeepBlanksDefault (0);
  doc = xmlParseFile (file_in);
  if (!doc)
    {
      g_error ("dtags-c: parsing XML tags file : doc == NULL");
      return;
    }
  cur = xmlDocGetRootElement (doc);
  if (!cur)
    {
      xmlFreeDoc (doc);
      g_error ("dtags-c: parsing XML tags file : cur == NULL");
      return;
    }
  cur = cur->xmlChildrenNode;
  while (cur)
    {
      name = xmlGetProp (cur, "name");
      g_hash_table_insert (tags_file, name, xmlGetProp (cur, "file"));
      g_hash_table_insert (tags_text, name, xmlGetProp (cur, "text"));
      g_hash_table_insert (tags_type, name, xmlGetProp (cur, "type"));
      g_hash_table_insert (tags_line, name, xmlGetProp (cur, "line"));
      cur = cur->next;
    }
  xmlFreeDoc (doc);
}

/**
 * string2id:
 * @value: value
 *
 * fr: Change '_' en '-'
 *
 * en: Changes '_' to '-'
 *
 * Return value: string adjusted
 **/

gchar* string2id (gchar *value)
{
  gchar *ret;
  ret = g_strdup (value);
  if (*ret == '_')
    ret = g_strndup (ret+1, strlen (ret)-1);
  while (strstr (ret, "_"))
    ret = string_replace (ret, "_", "-");
  g_strdown (ret);
  return (ret);
}

/**
 * string2name:
 * @value: value
 *
 * fr: Change 'gchar *td_app_custom_value' en 'td_app_custom_value'
 *
 * en: Changes 'gchar *td_app_custom_value' to 'td_app_custom_value'
 *
 * Return value: string adjusted
 **/

gchar* string2name (gchar *value)
{
  gchar *ret;
  gchar **string2;
  ret = g_strdup (value);
  string2 = g_strsplit (ret, " ", 2);
  if (string2[1])
    ret = g_strdup (string2[1]);
  if (*ret == '*')
    ret = g_strndup (ret+1, strlen (ret)-1);
  if (*ret == '_')
    ret = g_strndup (ret+1, strlen (ret)-1);
  return (ret);
}

/**
 * string2comment:
 * @value: value
 *
 * fr: Change '*** Message os command *' en 'Message os command'
 *
 * en: Changes '*** Message os command *' to 'Message os command'
 *
 * Return value: string adjusted
 **/

gchar* string2comment (gchar *value)
{
  gchar *ret;
  gchar **string2;
  ret = g_strdup (value);
  string2 = g_strsplit (ret, "/* ", 2);
  if (string2[1])
    ret = g_strdup (string2[1]);
  string2 = g_strsplit (ret, "/*** ", 2);
  if (string2[1])
    ret = g_strdup (string2[1]);
  string2 = g_strsplit (ret, " */", 2);
  if (string2[0])
    ret = g_strdup (string2[0]);
  return (ret);
}

/**
 * extract:
 * @file_out: output file location
 *
 * fr: Extraction
 *
 * en: Extract
 **/

void extract (gchar *file_out)
{
  FILE *fp;
  XmlDoc *doc;
  XmlNode *cur;
  XmlNode *cur2;
  GList *list_tmp;
  GList *list_tmp2;
  GHashTable *file;
  GString *source = NULL;
  GString *header = NULL;
  gchar **string2;
  gchar **string3;
  gchar **string4;
  gchar **string5;
  gchar **string6;
  gchar **string7;
  int i, j;
  GList *item;
  file = g_hash_table_new (g_str_hash, g_str_equal);
 
  /*** Chargement des fichiers source - Source files loading */
  list_tmp = hashtable_get_values (tags_file);
  for (i=0; i<g_list_length (list_tmp); i++)
    if (!g_hash_table_lookup (file, g_list_nth_data (list_tmp, i)))
      g_hash_table_insert (file, g_list_nth_data (list_tmp, i), g_list_nth_data (list_tmp, i));
  list_tmp = hashtable_get_values (file);
  if (g_list_length (list_tmp)<1)
    return;
  if (g_list_length (list_tmp)>1)
    {
      if (strstr (g_list_nth_data (list_tmp, 0), ".h"))
	{
	  header = string_file (g_list_nth_data (list_tmp, 0));
	  source = string_file (g_list_nth_data (list_tmp, 1));
	}
      else
	{
	  source = string_file (g_list_nth_data (list_tmp, 0));
	  header = string_file (g_list_nth_data (list_tmp, 1));
	}
    }
  else
    source = string_file (g_list_nth_data (list_tmp, 0));

  /*** Entte - Header */
  /* FIXME: description : +tards */ 
  doc = xmlNewDoc ("1.0");
  doc->xmlRootNode = xmlNewDocNode (doc, NULL, "dtags", NULL);
  string2 = g_strsplit (file_out, ".", 2);
  /* xmlSetProp (doc->xmlRootNode, "id", string2[0]); */
  if (strstr (source->str, "/**\n"))
    {
      cur = xmlNewChild (doc->xmlRootNode, NULL, "head", NULL);
      string2 = g_strsplit (source->str, "/**\n", 2);
      string3 = g_strsplit (string2[1], " **/\n", 2);

      /*** Membres - Members */
      string4 = g_strsplit (string3[0], "\n", strlen (string3[0]));
      i = 1;
      while (string4[i])
	{
	  string5 = g_strsplit (string4[i], ": ", 2);
  	  string6 = g_strsplit (string5[0], "@", 2);
	  if ((string6[1]) && (string5[1]))
	    xmlSetProp (cur, g_strstrip (string6[1]), g_strstrip (string5[1]));
	  i++;
	}

      /*** Description */
      string4 = g_strsplit (string3[0], "\n *\n * ", strlen (string3[0]));
      j=1;
      while (string4[j])
	{
	  cur2 = xmlNewChild (cur, NULL, "description", NULL);
	  string5 = g_strsplit (string4[j], "\n", 2);
	  xmlSetProp (cur2, "text", g_strstrip (string5[0]));
	  j++;
	}
    }

  /*** Macro */
  /* FIXME: description : +tards */ 
  list_tmp = hashtable_get_keys_type ("macro");
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      if (strstr (g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)), ".h"))
	string2 = g_strsplit (header->str, g_strdup_printf ("/**\n * %s:\n", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      else
	string2 = g_strsplit (source->str, g_strdup_printf ("/**\n * %s:\n", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string2[1])
	continue;
      string3 = g_strsplit (string2[1], " **/\n", 2);
    }

  /*** Enumration - Enumeration */
  list_tmp = hashtable_get_keys_type ("typedef");
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      if (strstr (g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)), ".h"))
	string2 = g_strsplit (header->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      else
	string2 = g_strsplit (source->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string2[1])
	continue;
      cur = xmlNewChild (doc->xmlRootNode, NULL, "enum", NULL);
      xmlSetProp (cur, "text", g_strstrip (g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "name", g_strstrip (string2name (g_list_nth_data (list_tmp, i))));
      xmlSetProp (cur, "id", g_strstrip (string2id (g_list_nth_data (list_tmp, i))));
      xmlSetProp (cur, "file", g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "line", g_hash_table_lookup (tags_line, g_list_nth_data (list_tmp, i)));

      /*** Membres - Members */
      string3 = g_strsplit (string2[1], "typedef enum {\n", 2);
      if (!string3[1])
	continue;
      string4 = g_strsplit (string3[1], g_strdup_printf ("} %s;\n", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string4[1])
	continue;
      string5 = g_strsplit (string4[0], "\n", strlen (string4[0]));
      j = 0;
      while (string5[j])
	{
	  string6 = g_strsplit (string5[j], ",", 2);
	  if ((string6[0]) && (string6[1]))
	    {
	      cur2 = xmlNewChild (cur, NULL, "member", NULL);
	      xmlSetProp (cur2, "name", g_strstrip (string6[0]));
	      xmlSetProp (cur2, "comment", g_strstrip (string2comment (string6[1])));
	    }
	  else
	    g_print ("dtags-c: member '%s.%s' not documented\n", (gchar*) g_list_nth_data (list_tmp, i), string5[j]);
	  j++;
	}

      /*** Description */
      string3 = g_strsplit (string2[1], " **/\n", 2);
      string4 = g_strsplit (string3[0], "\n *\n * ", strlen (string3[0]));
      j=1;
      while (string4[j])
	{
	  cur2 = xmlNewChild (cur, NULL, "description", NULL);
	  string5 = g_strsplit (string4[j], "\n", 2);
	  xmlSetProp (cur2, "text", g_strstrip (string5[0]));
	  j++;
	}
    }

  /*** Structure */
  list_tmp = hashtable_get_keys_type ("struct");
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      if (strstr (g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)), ".h"))
	string2 = g_strsplit (header->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      else
	string2 = g_strsplit (source->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string2[1])
	continue;
      cur = xmlNewChild (doc->xmlRootNode, NULL, "struct", NULL);
      xmlSetProp (cur, "text", g_strstrip (g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "name", g_strstrip (string2name (g_list_nth_data (list_tmp, i))));
      xmlSetProp (cur, "id", g_strstrip (string2id (g_list_nth_data (list_tmp, i))));
      xmlSetProp (cur, "file", g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "line", g_hash_table_lookup (tags_line, g_list_nth_data (list_tmp, i)));

      /*** Membres - Members */
      string3 = g_strsplit (string2[1], g_strdup_printf ("struct %s\n{", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string3[1])
	continue;
      string4 = g_strsplit (string3[1], g_strdup_printf ("};\n", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string4[1])
	continue;
      string5 = g_strsplit (string4[0], "\n", strlen (string4[0]));
      j = 0;
      while (string5[j])
	{
	  string6 = g_strsplit (string5[j], ";", 2);
	  if ((string6[0]) && (string6[1]))
	    {
	      cur2 = xmlNewChild (cur, NULL, "member", NULL);
	      xmlSetProp (cur2, "text", g_strstrip (string6[0]));
	      xmlSetProp (cur2, "name", g_strstrip (string2name (string6[0])));
	      xmlSetProp (cur2, "comment", g_strstrip (string2comment (string6[1])));
	    }
	  else
	    g_print ("dtags-c: member '%s.%s' not documented\n", (gchar*) g_list_nth_data (list_tmp, i), string5[j]);
	  j++;
	}

      /*** Description */
      string3 = g_strsplit (string2[1], " **/\n", 2);
      string4 = g_strsplit (string3[0], "\n *\n * ", strlen (string3[0]));
      j=1;
      while (string4[j])
	{
	  cur2 = xmlNewChild (cur, NULL, "description", NULL);
	  string5 = g_strsplit (string4[j], "\n", 2);
	  xmlSetProp (cur2, "text", g_strstrip (string5[0]));
	  j++;
	}
    }

  /*** Fonction - Function */
  list_tmp = hashtable_get_keys_type ("function");
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      if (strstr (g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)), ".h"))
	string2 = g_strsplit (header->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      else
	string2 = g_strsplit (source->str, g_strdup_printf ("/**\n * %s:", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string2[1])
	continue;
      if (string2[2])
	string2[1] = g_strdup (string2[2]);
      cur = xmlNewChild (doc->xmlRootNode, NULL, "function", NULL);
      xmlSetProp (cur, "name", g_strstrip (g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "id", g_strstrip (string2id (g_list_nth_data (list_tmp, i))));
      string3 = g_strsplit (string2[1], " **/\n", 2);
      string4 = g_strsplit (string3[1], g_strdup_printf ("%s (", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string4[0])
	continue;
      xmlSetProp (cur, "type", g_strstrip (string4[0]));
      xmlSetProp (cur, "file", g_hash_table_lookup (tags_file, g_list_nth_data (list_tmp, i)));
      xmlSetProp (cur, "line", g_hash_table_lookup (tags_line, g_list_nth_data (list_tmp, i)));

      /*** Arguments */
      string3 = g_strsplit (string2[1], g_strdup_printf ("%s (", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if (!string3[1])
	continue;
      string4 = g_strsplit (string3[1], g_strdup_printf (")\n{", (gchar*) g_list_nth_data (list_tmp, i)), 2);
      if ((!string4[0]) || (!string4[1]))
	continue;
      j = 0;
      string5 = g_strsplit (string4[0], ",", strlen (string4[0]));
      list_tmp2 = NULL;
      while (string5[j])
	{
	  list_tmp2 = g_list_append (list_tmp2, xmlNewChild (cur, NULL, "argument", NULL));
	  xmlSetProp (g_list_nth_data (list_tmp2, j), "text", g_strstrip (string5[j]));
	  j++;
	}
      string3 = g_strsplit (string2[1], " **/\n", 2);
      string4 = g_strsplit (string3[0], "\n *", strlen (string3[0]));
      j = 1;
      while (string4[j])
	{
	  if ((!strstr (string4[j], "@")) || (!strstr (string4[j], ":")))
	    {
	      j++;
	      continue;
	    }
	  string5 = g_strsplit (string4[j], "\n", 2);
	  string6 = g_strsplit (string5[0], "@", 2);
	  string7 = g_strsplit (string6[1], ":", 2);
	  if ((string7[0]) && (string7[1]) && (g_list_nth_data (list_tmp2, j-1)))
	    {
	      xmlSetProp (g_list_nth_data (list_tmp2, j-1), "name", g_strstrip (string7[0]));
	      xmlSetProp (g_list_nth_data (list_tmp2, j-1), "comment", g_strstrip (string2comment (string7[1])));
	    }
	  else
	    g_print ("dtags-c: member '%s.%s' not found\n", (gchar*) g_list_nth_data (list_tmp, i), (gchar*) g_list_nth_data (list_tmp2, j-1)); 
	  j++;
	}
      for (; j<g_list_length (list_tmp2); j++)
	g_print ("dtags-c: member '%s.%s' not found\n", (gchar*) g_list_nth_data (list_tmp, i), (gchar*) g_list_nth_data (list_tmp2, j-1)); 

      /*** Description */
      string4 = g_strsplit (string3[0], "\n *\n * ", strlen (string3[0]));
      j=1;
      while (string4[j])
	{
	  cur2 = xmlNewChild (cur, NULL, "description", NULL);
	  string5 = g_strsplit (string4[j], "\n", 2);
	  xmlSetProp (cur2, "text", g_strstrip (string5[0]));
	  j++;
	}
    }

  /*** Enregistrement - Save */
  fp = fopen (file_out, "w");
  if (!fp)
    {
      g_error ("dtags-c: opening XML file failed");
      xmlFreeDoc (doc);
      return;
    }
  fclose (fp);
  /* FIXME: libxml2 : attente d'une version plus mature - libxml2 : wait a more mature version */ 
  /* if (base->encoding) */
  /* { */
  /* if (xmlSaveFileEnc (file, doc, base->encoding)<0) */
  /* { */
  /* td_app_message (g_strdup_printf (_("Saving geometry '%s'"), file), TD_MSG_FAILED); */
  /* xmlFreeDoc (doc); */
  /* return FALSE; */
  /* } */
  /* } */
  /* else */
  if (xmlSaveFile (file_out, doc)<0)
    {
      g_error ("dtags-c: saving XML file failed");
      xmlFreeDoc (doc);
      return;
    }
  xmlFreeDoc (doc);
}

/*****************************************************************************/
/*** Principale - Main */
/*****************************************************************************/

/**
 * popts_about:
 *
 * fr: Affiche le  propos du programme
 *
 * en: Display the about of the program
 **/

void popts_about (void)
{
  g_print ("%s %s : Balises pour documentation - Tags for documentation\n", PACKAGE_COMMAND, DTAGS_VERSION);
  g_print ("Copyright (c) 2001 Philippe Roy\n");
  g_print ("Author : Philippe Roy <ph_roy@toutdoux.org>\n");
  g_print ("dtags-c is a free software, covered by the GNU General Public License. (http://www.gnu.org)\n");
  g_print ("dtags-c is a tool developped for the ToutDoux project. (http://www.toutdoux.org)\n");
  exit (0);
}

/**
 * popts_def:
 *
 * fr: Definition des options de la ligne de commande
 *
 * en: Definition of the options of command line
 **/

void popts_def (void)
{
  PACKAGE_COMMAND = "dtags-c";
  popts[0] = popt_def (NULL, "about", NULL, "About");
  popts[1] = popt_def ("v", "version", NULL, "Version");
  popts[2] = popt_def ("i", "input", "<xml tags file>", "Specify the XML input tags file (obligatory)");
  popts[3] = popt_def ("o", "output", "<xml file>", "Specify the XML output file (obligatory)");
};

/**
 * main:
 * @argc: counter of command line options used
 * @argv[]: arraw of command line options used
 *
 * fr: Procdure principale du programme
 *
 * en: Main procedure of the program
 *
 * Return value: -1 on error
 **/

int main (gint argc, char *argv[])
{
  FILE *file_in;
  FILE *file_out;
  void *ctx;

  /*** Popts */
  popts_def();
  popts_parse (argc, argv, 4, popts);

  /*** Vrification des fichiers - Files checking */
  file_in = fopen (popts[2]->value, "r");
  if (file_in == NULL)
    g_error ("dtags-c: reading XML input file : error in opening");
  fclose (file_in);
  file_out = fopen (popts[3]->value, "w");
  if (file_out == NULL)
    g_error ("dtags-c: reading XML output file : error in opening");
  fclose (file_out);

  /*** Extraction - Extracting */
  load_tags (popts[2]->value);
  extract (popts[3]->value);
  return 0;
}


