/**
 * td_app:
 * @title: TdApp
 * @subtitle: Application
 * @project: Toutdoux
 * @lang: fr,en
 * @authors: Philippe Roy <ph_roy@toutdoux.org>
 * @copyright: Copyright (c) 2000-2001 Philippe Roy
 * @license: GNU GPL
 *
 * fr: Object application
 *
 * en: Application object
 **/

/*
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <dlfcn.h>
#include <dirent.h>
#include <sys/stat.h>

#include "commons.h"
#include "gtdk.h"
#include "database.h"
#include "td_app.h"
#include "td_app_palette.h"
#include "td_app_bench.h"
#include "td_app_etabliste.h"
#include "td_app_command.h"
#include "td_about.h"
#include "td_mod.h"
#include "td_db_base.h"
#include "td_db_table.h"
#include "td_db_connect.h"
#include "td_function.h"
#include "td_flower.h"

/* 
 * Re: Pourquoi ceux qui aiment SuSE sont forcement des Trolls ?  (Score: 3 - humour)
 * Ajout par Anonyme - Samedi 16 Juin  22:07
 *
 * Bonjour
 *
 * le nom de ma socit est Lave'tout Rapide. C'est une laverie automatique 
 * spcialise dans les tissus dlicats.
 *
 * Avant j'etais sous Redhat, puis je suis pass sous Mandrake. 
 * Et, par experience, je peux vous affirmer que Mandrake c'est de la merde : toujours 
 * un probleme. De plus, Mandrake ne veut pas suivre les regles LSB, et a a me fait 
 * partir en vrille. De plus, leur support c'est rien du tout : je leur ai demand si 
 * ils pouvaient m'aider sur un pb de maintien des couleurs  60, ils m'ont dit que 
 * j'avais trop de competence.
 *
 * Puis maintenant je suis avec SuSE : je suis un de leur partenaire. SuSE, c'est bien 
 * parce que des qu'on a un pepin ils sont l. De plus, le support de lin et la 
 * conformit avec Woolmark sont l.
 *
 * Voila, et en plus, SuSE nous permet de faire des economie : pour 1000 SuSE achets, 
 * 100 barrils d'Ariel gratuits...
 *
 * LinuxFr
 */

/*****************************************************************************/
/*** Arguments */
/*****************************************************************************/

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_MODIFIED,
  ARG_MODE_ECHO,
};

static void td_app_set_modified (gboolean modified)
{
  if (TD_APP(APP)->modified != modified)
    TD_APP(APP)->modified = modified;
}

static void td_app_set_mode_echo (gint mode_echo)
{
  if (TD_APP(APP)->mode_echo != mode_echo)
    TD_APP(APP)->mode_echo = mode_echo;
}

static void td_app_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdApp *app;
  app = TD_APP (object);
  switch (arg_id)
    {
    case ARG_MODIFIED:
      td_app_set_modified (GTK_VALUE_BOOL (*arg));
      break;
    case ARG_MODE_ECHO:
      td_app_set_mode_echo (GTK_VALUE_INT (*arg));
      break;
    default:
      break;
    }
}

static void td_app_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdApp *app;
  app = TD_APP (object);
   switch (arg_id)
    {
    case ARG_MODIFIED:
      GTK_VALUE_BOOL (*arg) = app->modified;
      break;
    case ARG_MODE_ECHO:
      GTK_VALUE_INT (*arg) = app->mode_echo;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

/*****************************************************************************/
/*** Initialisation */
/*****************************************************************************/

static void td_app_init (TdApp *app)
{
  app->pid = 0;
  app->pid_database = NULL;
  app->palette = td_app_palette_new();
  app->bench = td_app_bench_new();
  app->etabliste = td_app_etabliste_new();
  app->command = td_app_command_new();
  app->modified = FALSE;
  app->mode_echo = TD_MODE_ECHO_BW;
  app->admin_field = NULL;
  app->connect = NULL;
  app->base = NULL;

  app->cmd = g_hash_table_new (g_str_hash, g_str_equal);
  app->cmd_cmp = g_completion_new (NULL);
  app->cmd_accel = gtk_accel_group_new();
  app->menu_main = NULL;
  app->menu_palette = NULL;
  app->menu_favorites = NULL;
  app->menu_templates = NULL;
  app->menu_commands = NULL;
  app->menu_history = NULL;

  app->about = td_about_new();
  app->custom = g_hash_table_new (g_str_hash, g_str_equal);
  app->mod_core = g_hash_table_new (g_str_hash, g_str_equal);
  app->mod_data = g_hash_table_new (g_str_hash, g_str_equal);
  app->mod_filter = g_hash_table_new (g_str_hash, g_str_equal);
  app->process = NULL;

  app->function = NULL;
}

static void td_app_class_init (TdAppClass *klass)
{
  GtkObjectClass *object_class;
  object_class = (GtkObjectClass*) klass;
  parent_class = gtk_type_class (gtk_object_get_type());
  gtk_object_add_arg_type ("TdApp::modified", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_MODIFIED);
  gtk_object_add_arg_type ("TdApp::mode_echo", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_MODE_ECHO);
  object_class->set_arg = td_app_set_arg;
  object_class->get_arg = td_app_get_arg;

  klass->add_admin_field = td_app_add_admin_field;
  klass->add_cmd = td_app_add_cmd;
  klass->set_menu_main = td_app_set_menu_main;
  klass->set_menu_palette = td_app_set_menu_palette;
  klass->set_menu_favorites = td_app_set_menu_favorites;
  klass->set_menu_templates = td_app_set_menu_templates;
  klass->set_menu_commands = td_app_set_menu_commands;
  klass->set_menu_history = td_app_set_menu_history;

  klass->set_current = td_app_set_current;
  klass->set_modified = td_app_set_modified;

  klass->add_function = td_app_add_function;
  klass->create = td_app_create;
  klass->quit = td_app_quit;
}

GtkType td_app_get_type (void)
{
  static GtkType app_type = 0;
  if (!app_type)
    {
      static const GtkTypeInfo app_info =
      {
  	"TdApp", sizeof (TdApp), sizeof (TdAppClass),
  	(GtkClassInitFunc) td_app_class_init,
	(GtkObjectInitFunc) td_app_init,
	NULL, NULL, (GtkClassInitFunc) NULL,
      };
      app_type = gtk_type_unique (GTK_TYPE_OBJECT, &app_info);
    }
  return app_type;
}

/**
 * td_app_new:
 *
 * fr: Cr une nouvelle application
 *
 * en: Creates a new application
 *
 * Return value: application
 **/

GtkObject* td_app_new (void)
{
  return GTK_OBJECT (gtk_type_new (td_app_get_type()));
}

/**
 * td_app_quit:
 * @app: Application
 *
 * fr: Dtruit l'application
 *
 * en: Destroys the application
 **/

void td_app_quit (TdApp *app)
{
  void (*symbol)() = NULL;
  gint i;
  GList *list_tmp;
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Fermeture du fichier en cours - Closing the current file */
  td_app_message (_("Stopping"), TD_MSG_BEGIN);
  td_app_message (_("Stopping"), TD_MSG_PROGRESS);
  if (td_db_connect_close())
    {

      /*** Historique - History */
/*        (gpointer*) symbol = td_dlsym (TD_PLUGINS_CORE_BOOKMARKS, NULL, "libbookmarks.so", "plugins_save_history", TRUE); */
/*        symbol(); */

      /*** Modules */
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = TRUE;
/*        list_tmp = td_hashtable_get_values (app->mod_core); */
/*        for (i=0; i<g_list_length (list_tmp); i++) */
/*  	td_mod_destroy (g_list_nth_data (list_tmp, i)); */
/*        list_tmp = td_hashtable_get_values (app->mod_filter); */
/*        for (i=0; i<g_list_length (list_tmp); i++) */
/*  	td_mod_destroy (g_list_nth_data (list_tmp, i)); */
/*        TD_PLUGINS_CORE_DATABASE = NULL; */
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = FALSE;
      td_app_message (_("Stopping"), TD_MSG_END);
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = TRUE;

      /*** Verrou - Lock */
      td_app_lock_remove (app->pid);
      td_database_base_logout();
      td_database_base_drop (app->pid_database);
      TD_PLUGINS_CORE_DATABASE = NULL;

      /*** Structure */
      app->pid = 0;
      app->pid_database = NULL;
      td_app_palette_destroy (TD_APP_PALETTE (app->palette));
      td_app_bench_destroy (TD_APP_BENCH (app->bench));
      td_app_etabliste_destroy (TD_APP_ETABLISTE (app->etabliste));
      td_app_command_destroy (TD_APP_COMMAND (app->command));
      app->mode_echo = 0;
      g_list_free (app->admin_field);
      app->modified = 0;
      if (app->connect)
	gtk_object_destroy (app->connect);
      if (app->base)
	gtk_object_destroy (app->base);

      g_hash_table_destroy (app->cmd);
      g_completion_free (app->cmd_cmp);
      /* gtk_widget_destroy (app->cmd_accel); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */
      if (app->menu_main)
	gtk_widget_destroy (app->menu_main);
      /* gtk_widget_destroy (app->menu_palette); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */
      /* gtk_widget_destroy (app->menu_favorites); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */
      /* gtk_widget_destroy (app->menu_templates); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */
      /* gtk_widget_destroy (app->menu_commands); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */
      /* gtk_widget_destroy (app->menu_history); *//*** CF: td_app_palette_destroy (TD_APP_PALETTE (app->palette) */

      td_about_destroy (TD_ABOUT (app->about));
      g_hash_table_destroy (app->custom);
      g_hash_table_destroy (app->mod_core);
      g_hash_table_destroy (app->mod_data);
      g_hash_table_destroy (app->mod_filter);
      g_list_free (app->process);

      g_list_free (app->function);
      g_list_free (app->function_input);

      /*** Objet - Object */
      gtk_object_destroy (GTK_OBJECT (app));

      /*** Sortie - Exit */
      td_exit (0);
    }
  td_app_message (_("Stopping"), TD_MSG_END);
}

/*****************************************************************************/
/*** Affectations - Allocations */
/*****************************************************************************/

/**
 * td_app_add_admin_field:
 * @field: field
 *
 * fr: Ajoute un champs d'administration  l'application
 *
 * en: Adds a admintration field to application
 **/

void td_app_add_admin_field (gchar *field)
{
  TD_APP(APP)->admin_field = g_list_append (TD_APP(APP)->admin_field, field);
}

/**
 * td_app_add_cmd:
 * @name: name
 * @function: function
 * @required: number of required arguments
 * @optional: number of optional args
 * @rest: flag for the 'rest' list
 *
 * fr: Ajoute une incantation  l'application
 *
 * en: Adds an incantation to application
 **/

void td_app_add_cmd (gchar *name, Scm (*function)(), int required, int optional, int rest)
{
  GList *list_tmp = NULL;
  g_hash_table_insert (TD_APP(APP)->cmd, td_strsplit (name, " ", 0), name);
  gh_new_procedure (td_strsplit (name, " ", 0), function, required, optional, rest);
  list_tmp = g_list_append (list_tmp, name);
  g_completion_add_items (TD_APP(APP)->cmd_cmp, list_tmp);
  g_list_free (list_tmp);
}

/**
 * td_app_set_menu_main:
 * @menu_main: menu
 *
 * fr: Affecte le menu comme menu principal  l'application
 *
 * en: Allocates the menu like main menu to application
 **/

void td_app_set_menu_main (GtkWidget *menu_main)
{
  if (TD_APP(APP)->menu_main != menu_main)
    TD_APP(APP)->menu_main = menu_main;
}

/**
 * td_app_set_menu_palette:
 * @menu_palette: menu
 *
 * fr: Affecte le menu comme menu de palette  l'application
 *
 * en: Allocates the menu like palette menu to application
 **/

void td_app_set_menu_palette (GtkWidget *menu_palette)
{
  if (TD_APP(APP)->menu_palette != menu_palette)
    TD_APP(APP)->menu_palette = menu_palette;
}

/**
 * td_app_set_menu_favorites:
 * @menu_favorites: menu
 *
 * fr: Affecte le menu comme menu des favoris  l'application
 *
 * en: Allocates the menu like favorites menu to application
 **/

void td_app_set_menu_favorites (GtkWidget *menu_favorites)
{
  if (TD_APP(APP)->menu_favorites != menu_favorites)
    TD_APP(APP)->menu_favorites = menu_favorites;
}

/**
 * td_app_set_menu_templates:
 * @menu_templates: menu
 *
 * fr: Affecte le menu comme menu des gabarits  l'application
 *
 * en: Allocates the menu like templates menu to application
 **/

void td_app_set_menu_templates (GtkWidget *menu_templates)
{
  if (TD_APP(APP)->menu_templates != menu_templates)
    TD_APP(APP)->menu_templates = menu_templates;
}

/**
 * td_app_set_menu_commands:
 * @menu_commands: menu
 *
 * fr: Affecte le menu comme menu des commandes  l'application
 *
 * en: Allocates the menu like commands menu to application
 **/

void td_app_set_menu_commands (GtkWidget *menu_commands)
{
  if (TD_APP(APP)->menu_commands != menu_commands)
    TD_APP(APP)->menu_commands = menu_commands;
}

/**
 * td_app_set_menu_history:
 * @menu_history: menu
 *
 * fr: Affecte le menu comme menu de l'historique  l'application
 *
 * en: Allocates the menu like history menu to application
 **/

void td_app_set_menu_history (GtkWidget *menu_history)
{
  if (TD_APP(APP)->menu_history != menu_history)
    TD_APP(APP)->menu_history = menu_history;
}

/**
 * td_app_set_current:
 * @connect: connection
 * @base: base
 *
 * fr: Affecte la connection et la base  l'application
 *
 * en: Allocates the connection and database to application
 **/

void td_app_set_current (GtkObject *connect, GtkObject *base)
{
  gint i, j, k;
  GtkObject *function;
  TdFunction *cur_function;

  /*** Base */
  if (TD_APP(APP)->connect != connect)
    TD_APP(APP)->connect = connect;
  if (TD_APP(APP)->base != base)
    TD_APP(APP)->base = base;
  if (TD_DB_CONNECT (connect)->type != TD_CONNECT_TYPE_NONE)
    td_app_lock_update (TD_DB_CONNECT (connect)->file);

  /*** Fonction - Function */
  if (base)
    for (i=0; i<g_list_length (TD_DB_BASE (base)->table); i++)
      for (j=0; j<g_list_length (TD_DB_TABLE (g_list_nth_data (TD_DB_BASE (base)->table, i))->function); j++)
	{
	  cur_function = TD_FUNCTION (g_list_nth_data (TD_DB_TABLE (g_list_nth_data (TD_DB_BASE (base)->table, i))->function, j));
	  function = td_function_new();
	  gtk_object_set (GTK_OBJECT (function),
			  "name", cur_function->name,
			  "comment", cur_function->comment, NULL);
	  for (k=0; k<g_list_length (cur_function->property_name); k++)
	    {
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "input"))
		{
		  td_function_add_input (TD_FUNCTION (function), (gchar*) g_list_nth_data (cur_function->property_value, k));
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "output"))
		{
		  td_function_add_output (TD_FUNCTION (function), (gchar*) g_list_nth_data (cur_function->property_value, k));
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "language"))
		{
		  gtk_object_set (GTK_OBJECT (function), "language", (gchar*) g_list_nth_data (cur_function->property_value, k), NULL);
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "definition"))
		{
		  gtk_object_set (GTK_OBJECT (function), "definition", (gchar*) g_list_nth_data (cur_function->property_value, k), NULL);
		  continue;
		}
	    }
	  td_app_add_function (function);
	}
}

/**
 * td_app_add_mod:
 * @mod: module
 * @init: enable initialization
 *
 * fr: Ajoute un module  l'application
 *
 * en: Adds a module to application
 *
 * Return value: FALSE on error
 **/

gboolean td_app_add_mod (GtkObject *mod, gboolean init)
{
  void (*symbol)();

  /*** Dj charg - Already loaded */
  switch (TD_MOD (mod)->type)
    {
    case TD_MOD_TYPE_CORE:
      if (g_hash_table_lookup (TD_APP(APP)->mod_core, TD_MOD (mod)->name))
	return TRUE;
      break;
    case TD_MOD_TYPE_DATA:
      if (g_hash_table_lookup (TD_APP(APP)->mod_data, TD_MOD (mod)->name))
	return TRUE;
      break;
    case TD_MOD_TYPE_FILTER:
      if (g_hash_table_lookup (TD_APP(APP)->mod_filter, TD_MOD (mod)->name))
	return TRUE;
      break;
    default:
      break;
    }

  /*** Affectation - Allocation */
  switch (TD_MOD (mod)->type)
    {
    case TD_MOD_TYPE_CORE:
      g_hash_table_insert (TD_APP(APP)->mod_core, TD_MOD (mod)->name, mod);
      break;
    case TD_MOD_TYPE_DATA:
      g_hash_table_insert (TD_APP(APP)->mod_data, TD_MOD (mod)->name, mod);
      break;
    case TD_MOD_TYPE_FILTER:
      g_hash_table_insert (TD_APP(APP)->mod_filter, TD_MOD (mod)->name, mod);
      break;
    default:
      break;
    }

  /*** Initialisation - Initialization */
  if (init)
    {
      td_app_message (g_strdup_printf (_("Executing symbol '%s::plugins_init'"), TD_MOD (mod)->name), TD_MSG_PLUGINS);
      symbol = dlsym (TD_MOD (mod)->plugin, "plugins_init");
      if (dlerror() != NULL)
	{
	  td_app_message (g_strdup_printf (_("Executing symbol '%s::plugins_init' : not found"), TD_MOD (mod)->name), TD_MSG_FAILED);
	  return FALSE;
	}
      symbol (TD_MOD (mod));
    }

  /*** Etablis - Bench */
  if (TD_MOD (mod)->type == TD_MOD_TYPE_DATA)
    {
      td_app_palette_add_mod (TD_APP_PALETTE (TD_APP (APP)->palette), mod);
      td_app_bench_add_mod (TD_APP_BENCH (TD_APP (APP)->bench), mod);
      td_app_etabliste_add_mod (TD_APP_ETABLISTE (TD_APP (APP)->etabliste), mod);
    }

  /*** Visualisation - Visualization */
  if (td_mod_get ("xts", TD_MOD_TYPE_CORE))
    td_app_cmd (NULL, "xts-update");
  return TRUE;
}

/**
 * td_app_add_process:
 * @process: process
 *
 * fr: Ajoute un processus  l'application
 *
 * en: Adds a process to application
 **/

void td_app_add_process (GtkObject *process)
{
  TD_APP(APP)->process = g_list_append (TD_APP(APP)->process, process);
}

/**
 * td_app_add_function:
 * @function: function
 *
 * fr: Ajoute la fonction  l'application
 *
 * en: Adds the function to application
 **/

void td_app_add_function (GtkObject *function)
{
  gint i, j;
  gboolean bool_tmp;

  /*** Fonction - Function */
  TD_APP(APP)->function = g_list_append (TD_APP(APP)->function, function);
  
  /*** Champs - Field */
  for (i=0; i<g_list_length (TD_FUNCTION (function)->input); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (TD_APP(APP)->function_input); j++)
	if (!strcmp ((gchar*) g_list_nth_data (TD_FUNCTION (function)->input, i), (gchar*) g_list_nth_data (TD_APP(APP)->function_input, j)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if (bool_tmp)
	TD_APP(APP)->function_input = g_list_append (TD_APP(APP)->function_input, (gchar*) g_list_nth_data (TD_FUNCTION (function)->input, i));
    }
}

/*****************************************************************************/
/*** Commandes - Commands */
/*****************************************************************************/

/**
 * td_app_create:
 *
 * fr: Cr l'interface de l'application
 *
 * en: Creates the interface of application
 **/

void td_app_create (void)
{
  GtkObject *connect;
  void *module_tmp;
  void (*symbol_tmp)();
  GList *value_list;
  gint i;
  gint echo_mode;
  void (*symbol)();

  /*** Dorsal SQL - SQL back-end */
  if (!strcmp (td_app_custom_value ("database", "backend", "database-management-system"), "mysql"))
    {
      TD_PLUGINS_CORE_DATABASE = dlopen (g_strdup_printf ("%s%s", TD_PACKAGE_PLUGINS_DIR, "libmysql.so"), RTLD_LAZY);
      if (dlerror() != NULL)
	td_app_message (g_strdup_printf (_("Loading SQL interface : file '%s%s' unreadeable"), TD_PACKAGE_PLUGINS_DIR, "libmysql.so"), TD_MSG_FAILED);
    }
  else
    {
      TD_PLUGINS_CORE_DATABASE = dlopen (g_strdup_printf ("%s%s", TD_PACKAGE_PLUGINS_DIR, "libpostgresql.so"), RTLD_LAZY);
      if (dlerror() != NULL)
	td_app_message (g_strdup_printf (_("Loading SQL interface : file '%s%s' unreadeable"), TD_PACKAGE_PLUGINS_DIR, "libpostgresql.so"), TD_MSG_FAILED);
    }
  connect = td_db_connect_new();
  gtk_object_set (GTK_OBJECT (connect), "name", _("New"), NULL);
  td_db_connect_set_type (TD_DB_CONNECT (connect), TD_CONNECT_TYPE_NONE);
  td_app_set_current (connect, NULL);
  td_app_lock_add();

  /*** Fentres de base - Basis windows */
  if (TD_APP(APP)->palette)
    {
      td_app_palette_create (TD_APP_PALETTE (TD_APP(APP)->palette));
      if (gtdk_geometry_get ("palette", "visible") == 1)
	td_app_palette_show (TD_APP_PALETTE (TD_APP(APP)->palette));
    }
  if (TD_APP(APP)->bench)
    {
      td_app_bench_create (TD_APP_BENCH (TD_APP(APP)->bench));
      if (gtdk_geometry_get ("bench", "visible") == 1)
	td_app_bench_show (TD_APP_BENCH (TD_APP(APP)->bench));
    }
  if (TD_APP(APP)->etabliste)
    {
      td_app_etabliste_create (TD_APP_ETABLISTE (TD_APP(APP)->etabliste));
      if (gtdk_geometry_get ("etabliste", "visible") == 1)
	td_app_etabliste_show (TD_APP_ETABLISTE (TD_APP(APP)->etabliste));
    }
  if (TD_APP(APP)->command)
    {
      td_app_command_create (TD_APP_COMMAND (TD_APP(APP)->command));
      if (gtdk_geometry_get ("command", "visible") == 1)
	td_app_command_show (TD_APP_COMMAND (TD_APP(APP)->command));
    }
}

/**
 * td_app_message:
 * @message: text
 * @type: type
 *
 * fr: Affiche un message sur la palette et dans la fentre des messages
 *
 * en: Displays message for palette's statusbar and message window
 **/

void td_app_message (gchar *message, TdMsg type)
{
  if (!message)
    return;

  /*** Shunt */
  if ((APP) &&
      ((TD_APP (APP)->mode_echo != TD_MODE_ECHO_ALL_COLORS) && (TD_APP (APP)->mode_echo != TD_MODE_ECHO_ALL_BW)) &&
      (((type == TD_MSG_GUILE) || (type == TD_MSG_OS) || type == TD_MSG_PLUGINS) || (type == TD_MSG_SQL_CACHE) || (type == TD_MSG_SQL) || (type == TD_MSG_FUNCTION)))
    return;

  /*** Guile */
  if (type == TD_MSG_GUILE)
    {
      if ((TD_APP (APP)->mode_echo == TD_MODE_ECHO_COLORS) || 
	  (TD_APP (APP)->mode_echo == TD_MODE_ECHO_ALL_COLORS))
	g_print ("\033[32m\033[1m%s : guile\033[0m\n", message);
      else
	g_print ("%s : guile\n", message);
      return;
    }

  /*** Message */
  if (TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE)
    {
      if (TD_FLAG_DEBUG_MODE)
	g_print ("%s : %s\n", message, _("hidden"));
      TD_START_MSG = g_list_append (TD_START_MSG, message);
      TD_START_MSG = g_list_append (TD_START_MSG, itop (type));
    }
  else
    gh_eval_str (g_strdup_printf ("(messages-add \"%s\" %d)", message, type));

  /*** Stop */
  if (type == TD_MSG_STOP)
    td_exit (1);
}

/**
 * td_app_action_function:
 * @table: table
 * @mod: module
 *
 * fr: Execute une modification gnre par une fonction
 *
 * en: Executes a modification generated by function
 **/

void td_app_action_function (gchar *table, GtkObject *mod)
{
/*    gint i, j; */
/*    gboolean bool_tmp; */
/*    TdFunction *function_cur; */
/*    void (*symbol)() = NULL; */

  /*** Entre d'une fonction ? - Function input ? */
/*    bool_tmp = FALSE; */
/*    for (i=0; i<g_list_length (TD_APP (APP)->function_input); i++) */
/*      if (!strcmp ((gchar*) g_list_nth_data (TD_APP (APP)->function_input, i), table)) */
/*        { */
/*  	bool_tmp = TRUE; */
/*  	break; */
/*        } */
/*    if (!bool_tmp) */
/*      return; */

  /*** Go ! */
/*    for (i=0; i<g_list_length (TD_APP (APP)->function); i++) */
/*      for (j=0; j<g_list_length (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i))->input); j++) */
/*        if (!strcmp ((gchar*) g_list_nth_data (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i))->input, j), table)) */
/*  	{ */
/*  	  if (!td_function_activate (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i)))) */
/*  	    bool_tmp = FALSE; */
/*  	  break; */
/*  	} */

  /*** Mise  jour - Update */
/*    if (!bool_tmp) */
/*      return; */
/*    (gpointer*) symbol = td_app_mod_data_symbol (TD_MOD (mod)->id, "plugins_function_action", TRUE); */
/*    if (symbol) */
/*      symbol (mod); */
}

/**
 * td_app_mode_echo_parse:
 * @value: string
 *
 * fr: Retourne le 'TdModeEcho' correspondant  la chane de caractres
 *
 * en: Returns the 'TdModeEcho' corresponding the string
 *
 * Return value: mode echo
 **/

TdModeEcho td_app_mode_echo_parse (gchar *value)
{
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "no"))
    return TD_MODE_ECHO_NO;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "colors"))
    return TD_MODE_ECHO_COLORS;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "all, colors"))
    return TD_MODE_ECHO_ALL_COLORS;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "black/white"))
    return TD_MODE_ECHO_BW;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "all, black/white"))
    return TD_MODE_ECHO_ALL_BW;
  return TD_MODE_ECHO_0;
}

/*****************************************************************************/
/*** Paramtrage - Parameters */
/*****************************************************************************/

/**
 * td_app_custom_load:
 *
 * fr: Charge la personnalisation
 *
 * en: Load the customize values
 **/

void td_app_custom_load (void)
{
  gchar *value;
  gchar *file;
  gchar *file_recovery;

  /*** Fichier - File */
  file = g_strdup_printf ("%s/customize/%s.xml", TD_PACKAGE_VAR_DIR, "app");
  if (!td_file_exist (file))
    {
      file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", _("en"));
      if (!td_file_exist (file))
	{
	  file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", "en");
	  if (!td_file_exist (file))
	    return;
	}
    }
  if (td_custom_load (TD_APP(APP)->custom, file))
    return;

  /*** Fichier standard - Standard file */
  if (strcmp (file, g_strdup_printf ("%s/customize/%s.xml", TD_PACKAGE_VAR_DIR, "app")))
    {
      td_app_message (g_strdup_printf (_("Loading customize '%s' : standard file corrupted"), file), TD_MSG_FAILED);
      return;
    }
  else
    {
      file_recovery = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", _("en"));
      if (!td_file_exist (file_recovery))
	{
	  file_recovery = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", "en");
	  if (!td_file_exist (file_recovery))
	    {
	      td_app_message (g_strdup_printf (_("Loading customize '%s' : standard file not found"), file), TD_MSG_FAILED);
	      return;
	    }
	}
    }
  if (!td_file_backup (file))
    {
      td_app_message (g_strdup_printf (_("Loading customize '%s' : create backup failed"), file), TD_MSG_FAILED);
      return;
    }
  if (td_custom_load (TD_APP(APP)->custom, file))
    return;
  td_app_message (g_strdup_printf (_("Loading customize '%s' : standard file corrupted"), file), TD_MSG_FAILED);
}

/**
 * td_app_custom_value:
 * @category: category
 * @group: group
 * @option: option
 *
 * fr: Retourne la valeur de l'option de personnalisation de l'application
 *
 * en: Returns the value of the option of the application's customize
 *
 * Return value: string
 **/

gchar* td_app_custom_value (gchar *category, gchar *group, gchar *option)
{
  gchar *ret;
  gchar *file;

  /*** Valeur personnalise - Customized value */
  ret = g_hash_table_lookup (TD_APP(APP)->custom, g_strdup_printf ("%s,%s,%s", category, group, option));
  if (ret)
    return ret;

  /*** Valeur standard - Standard value */
  file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", _("en"));
  if (!td_file_exist (file))
    {
      file = g_strdup_printf ("%s/share/%s/%s/customize-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", "en");
      if (!td_file_exist (file))
	{
	  td_app_message (g_strdup_printf (_("Loading customize '%s' : standard file not found"), file), TD_MSG_FAILED);
	  return NULL;
	}
    }
  td_custom_update (TD_APP(APP)->custom, file);
  return (g_hash_table_lookup (TD_APP(APP)->custom, g_strdup_printf ("%s,%s,%s", category, group, option)));
}

/**
 * td_app_cmd_load:
 *
 * fr: Charge les commandes (3 niveaux maxi et 50 menu maxi)
 *
 * en: Load the commands (3 levels max and 50 menus max)
 **/

void td_app_cmd_load (void)
{
  GtkObject *flower;
  GtkWidget *menu[50];
  GtkWidget *submenu[10];
  GtkWidget *subsubmenu[10];
  int i, j, k, m;
  gchar *path;
  GList *path_list;
  GHashTable *path_hash;
  gchar *file;
  void (*symbol)();

  /*** Fichier - File */
  file = g_strdup_printf ("%s/share/%s/%s/commands-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", _("en"));
  if (!td_file_exist (file))
    {
      file = g_strdup_printf ("%s/share/%s/%s/commands-%s.xml", TD_PACKAGE_DATA_DIR, TD_PACKAGE_DIR, "app", "en");
      if (!td_file_exist (file))
	{
	  td_app_message (g_strdup_printf (_("Loading commands '%s' : standard file not found"), file), TD_MSG_FAILED);
	  return;
	}
    }
  flower = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower),
		  "root", "commands",
		  "branch", "command",
		  "mode_type", TRUE, NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower), "name");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "name_intl");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "key");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "menu");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "menu_icon");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "menu_type");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "toolbar");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "toolbar_icon");
  td_flower_add_branch_attribut (TD_FLOWER (flower), "toolbar_type");
  if (!td_flower_load_file (TD_FLOWER (flower), file))
    {
      td_app_message (g_strdup_printf (_("Loading commands '%s' : file unreadable"), file), TD_MSG_FAILED);
      return;
    }

  /*** Menus */
  j=0;
  k=-1;
  m=-1;
  submenu[j] = gtk_menu_bar_new();
  td_app_set_menu_palette (submenu[j]);
  path_hash = g_hash_table_new (g_str_hash, g_str_equal);
  for (i=0; i<g_list_length (TD_FLOWER (flower)->items_value); i++)
    {
      if (strlen (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 3)) == 0)
	continue;
      path_list = td_strsplist (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 3), "/");
      path = g_strdup_printf ("%s/%s", (gchar*) g_list_nth_data (path_list, 0), (gchar*) g_list_nth_data (path_list, 1));
      
      /*** Niveau 0 - Level 0 */
      if ((g_list_nth_data (path_list, 0)) && (!g_hash_table_lookup (path_hash, g_list_nth_data (path_list, 0))))
	{
	  /*** Item */
	  m++;
	  menu[m] = gtk_menu_item_new_with_label (g_list_nth_data (path_list, 0));
	  gtk_menu_bar_append (GTK_MENU_BAR (submenu[0]), menu[m]);
	      
	  /*** Sous-menu - Submenu */
	  j++;
	  submenu[j] = gtk_menu_new();
	  gtdk_menu_item_set_submenu (GTK_MENU_ITEM (menu[m]), submenu[j]);
	  g_hash_table_insert (path_hash, g_list_nth_data (path_list, 0), submenu[j]);
	}

      /*** Niveau 1 - Level 1 */
      if ((g_list_nth_data (path_list, 1)) && (!g_hash_table_lookup (path_hash, path)))
	{
	  if (strstr (g_list_nth_data (path_list, 1), "::")) /*** Gnr - Generated */
	    {
	      /* FIXME:   : +tards */
/*  	      (gpointer*) symbol = td_dlsym (TD_PLUGINS_CORE_BOOKMARKS, NULL,  */
/*  					     td_strsplit (g_list_nth_data (path_list, 1), "::", 0),  */
/*  					     td_strsplit (g_list_nth_data (path_list, 1), "::", 1), TRUE); */
/*  	      symbol (GTK_MENU (g_hash_table_lookup (path_hash, g_list_nth_data (path_list, 0))), NULL); */
	    }
	  else
	    {
	      if (strstr (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 5), "+"))
		gtk_menu_append (GTK_MENU (g_hash_table_lookup (path_hash, g_list_nth_data (path_list, 0))), gtk_menu_item_new());
	      m++;
	      menu[m] = gtdk_menu_append (GTK_MENU (submenu[j]),
					  g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 0), 
					  NULL,
					  g_list_nth_data (path_list, 1),
					  g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 4),
					  TD_APP(APP)->cmd_accel,
					  gtdk_keyval_parse (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 2)),
					  gtdk_keymask_parse (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 2)));
	    }

	  /*** Sous-menu - Submenu */
	  if (g_list_nth_data (path_list, 2))
	    {
	      k++;
	      subsubmenu[k] = gtk_menu_new();
	      gtdk_menu_item_set_submenu (GTK_MENU_ITEM (menu[m]), subsubmenu[k]);
	      g_hash_table_insert (path_hash, path, subsubmenu[k]);
	    }
	  else
	    g_hash_table_insert (path_hash, path, menu[m]);
	}

      /*** Niveau 2 - Level 2 */
      if ((g_list_nth_data (path_list, 2)) && (strcmp ((g_list_nth_data (path_list, 2)), "(null)")))
	{
	  if (strstr (g_list_nth_data (path_list, 2), "::")) /*** Gnr - Generated */
	    {
	      /* FIXME:   : +tards */
/*  	      (gpointer*) symbol = td_dlsym (TD_PLUGINS_CORE_BOOKMARKS, NULL,  */
/*  					     td_strsplit (g_list_nth_data (path_list, 2), "::", 0),  */
/*  					     td_strsplit (g_list_nth_data (path_list, 2), "::", 1), TRUE); */
/*  	      symbol (GTK_MENU (g_hash_table_lookup (path_hash, path)), NULL); */
	    }
	  else
	    {
	      if (strstr (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 5), "+"))
		gtk_menu_append (GTK_MENU (g_hash_table_lookup (path_hash, g_list_nth_data (path_list, 0))), gtk_menu_item_new());
	      gtdk_menu_append (GTK_MENU (g_hash_table_lookup (path_hash, path)),
				g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 0), 
				NULL,
				g_list_nth_data (path_list, 2),
				g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 4),
				TD_APP(APP)->cmd_accel,
				gtdk_keyval_parse (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 2)),
				gtdk_keymask_parse (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower)->items_value, i), 2)));
	    }
	}
    }
  gtk_widget_show_all (submenu[0]);
  g_hash_table_destroy (path_hash);
  td_flower_destroy (TD_FLOWER (flower));
}

/**
 * td_app_cmd:
 * @widget: not used
 * @command: command
 *
 * fr: Execute la commande
 *
 * en: Execute the command
 **/

void td_app_cmd (GtkWidget *widget, gchar *command)
{

  /*** Nom - Name */
  if ((!command) || (strlen (command)==0))
    {
      td_app_message ("Command null", TD_MSG_FAILED);
      return;
    }

  /*** Commande - Command */
  if (*command != '(')
    {
      if (!g_hash_table_lookup (TD_APP(APP)->cmd, td_strsplit (command, " ", 0)))
	td_app_message (g_strdup_printf (_("Command '%s' not found"), command), TD_MSG_FAILED);
      else
	{
	  td_app_message (g_strdup_printf (_("Evaluating '(%s)'"), command), TD_MSG_GUILE);
	  gh_eval_str (g_strdup_printf ("(%s)", command));
	}
      return;
    }

  /*** Evaluation */
  td_app_message (g_strdup_printf (_("Evaluating '%s'"), command), TD_MSG_GUILE);
  gh_eval_str (command);
}

/*****************************************************************************/
/*** Verrou - Lock */
/*****************************************************************************/

/**
 * td_app_lock_add:
 *
 * fr: Ajoute le verrou de base de donnes
 *
 * en: Adds the database lock
 **/

static void td_app_lock_add (void)
{
  gint i, j;
  DIR *proc;
  struct dirent *entry;
  struct stat statb;
  GString *cmdline;
  gchar *locks_file;
  GtkObject *flower_new;
  GtkObject *flower_old;
  GList *list_tmp = NULL;
  gboolean bool_tmp;
  gint crash = -1;
  gint response = 0;
  gchar *txt_tmp;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);

  /*** Pid */
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    {
      td_app_message (g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
      TD_APP(APP)->pid == -1;
      return;
    }
  td_app_message (g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_DONE);
  proc = opendir ("/proc");
  if (!proc)
    {
      td_app_message (g_strdup_printf (_("Loading locks '%s' : processes file system not found"), locks_file), TD_MSG_FAILED);
      TD_APP(APP)->pid == -1;
      return;
    }
  for (entry = readdir (proc); entry; entry = readdir (proc))
    {
      if ((atoi (entry->d_name) == 0) || (stat (g_strdup_printf ("/proc/%s", entry->d_name), &statb)<0) || (!S_ISDIR (statb.st_mode)))
	continue;
      cmdline = td_string_file (g_strdup_printf ("/proc/%s/cmdline", entry->d_name));
      if (!cmdline)
	continue;
      if (strstr (cmdline->str, "toutdoux"))
	list_tmp = g_list_append (list_tmp, entry->d_name);
    }
  closedir (proc);
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (TD_FLOWER (flower_old)->items_value); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, j), 0), g_list_nth_data (list_tmp, i)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if (bool_tmp)
	{
	  TD_APP(APP)->pid = atoi (g_list_nth_data (list_tmp, i));
	  break;
	}
    }
  if (TD_APP(APP)->pid == 0)
    {
      td_app_message (g_strdup_printf (_("Loading locks '%s' : process not found"), locks_file), TD_MSG_FAILED);
      TD_APP(APP)->pid == -1;
      return;
    }

  /*** Crash ? */
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (list_tmp); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0), g_list_nth_data (list_tmp, j)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if ((bool_tmp) && (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 3), g_get_user_name())))
	{
	  crash = i;
	  break;
	}
    }
  response = 2;
  if (crash > -1)
    {
      td_app_message (g_strdup_printf (_("Crash detected at pid '%s' for the database '%s' with file '%s'"),
				       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0),
				       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1),
				       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2)), TD_MSG_MESSAGE);
      if ((!strcmp (td_app_custom_value ("database", "backend", "recovery"), "never")) || (TD_FLAG_DEBUG_MODE))
	response = 1;
      else
	{
	  if (!strcmp (td_app_custom_value ("database", "backend", "recovery"), "ask"))
	    response = td_question_create_window (g_strdup_printf (_("ToutDoux appears to have crashed or been killed last time it was run.\n Would you like to recover the state of your last session ?\n process %s : database '%s' with file '%s'\n "),
								   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0),
								   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1),
								   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2)), _("Warning"));
	  else
	    response = 0;
	}
    }

  /*** Rcupration - Recovery */
  if (response == 0)
    {
      td_app_message (_("Recovering the last session"), TD_MSG_MESSAGE);
      TD_APP(APP)->pid_database = g_strdup (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1));
      td_app_lock_remove (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0)));
      td_database_base_login (TD_APP(APP)->pid_database);
      gtk_object_set (GTK_OBJECT (TD_APP(APP)->connect),
		      "name", g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2),
		      "file", g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2), NULL);
      td_db_connect_set_type (TD_DB_CONNECT (TD_APP(APP)->connect), TD_CONNECT_TYPE_RECOVERY_START);
      td_flower_destroy (TD_FLOWER (flower_old));
      return;
    }
  
  /*** Destruction - Destroy */
  if (response == 1)
    {
      td_app_message (_("Destroying the last session"), TD_MSG_MESSAGE);
      TD_APP(APP)->pid_database = g_strdup (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1));
      td_app_lock_remove (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0)));
      td_database_base_drop (TD_APP(APP)->pid_database);
      td_database_base_create (TD_APP(APP)->pid_database);
      td_database_base_login (TD_APP(APP)->pid_database);
      td_flower_destroy (TD_FLOWER (flower_old));
      return;
    }

  /*** Nouvelle base - New base */
  i=0;
  bool_tmp = FALSE;
  while (!bool_tmp)
    {
      bool_tmp = TRUE;
      txt_tmp = g_strdup_printf ("%s%d", td_app_custom_value ("database", "backend", "prefix"), i++);
      for (j=0; j<g_list_length (TD_FLOWER (flower_old)->items_value); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, j), 1), txt_tmp))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
    }
  td_flower_destroy (TD_FLOWER (flower_old));
  TD_APP(APP)->pid_database = txt_tmp;
  td_app_lock_update ("");
  td_database_base_create (TD_APP(APP)->pid_database);
  td_database_base_login (TD_APP(APP)->pid_database);
}

/**
 * td_app_lock_update:
 * @file: file
 *
 * fr: Rafrachis le verrou de base de donnes
 *
 * en: Updates the database lock
 **/

static void td_app_lock_update (gchar *file)
{
  gint i;
  GtkObject *flower_new;
  GtkObject *flower_old;
  GList *list_tmp;
  gchar *locks_file;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);
  if (TD_APP(APP)->pid < 0)
    return;
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    {
      td_app_message (g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
      return;
    }
  flower_new = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_new), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "user");
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    if (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0)) != TD_APP(APP)->pid)
      td_flower_add_item (TD_FLOWER (flower_new), g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0);
  list_tmp = NULL;
  list_tmp = g_list_append (list_tmp, td_itoa (TD_APP(APP)->pid));
  list_tmp = g_list_append (list_tmp, TD_APP(APP)->pid_database);
  if (file)
    list_tmp = g_list_append (list_tmp, file);
  else
    list_tmp = g_list_append (list_tmp, "");
  list_tmp = g_list_append (list_tmp, g_get_user_name());
  td_flower_add_item (TD_FLOWER (flower_new), list_tmp, 0);
  if (!td_flower_save_file (TD_FLOWER (flower_new), locks_file))
    td_app_message (g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_DONE);
  td_flower_destroy (TD_FLOWER (flower_old));
  td_flower_destroy (TD_FLOWER (flower_new));
}

/**
 * td_app_lock_remove:
 * @pid: process identifier
 *
 * fr: Supprime le verrou de base de donnes
 *
 * en: Removes the database lock
 **/

static void td_app_lock_remove (gint pid)
{
  gint i;
  GString *cmdline;
  gchar *locks_file;
  GtkObject *flower_new;
  GtkObject *flower_old;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);
  if (pid<0)
    return;
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    td_app_message (g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_DONE);
  flower_new = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_new), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "user");
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    if (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0)) != pid)
      td_flower_add_item (TD_FLOWER (flower_new), g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0);
  if (!td_flower_save_file (TD_FLOWER (flower_new), locks_file))
    td_app_message (g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_DONE);
  td_flower_destroy (TD_FLOWER (flower_old));
  td_flower_destroy (TD_FLOWER (flower_new));
}
